package com.ejie.ab18a.webservice;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.zip.ZipInputStream;

import javax.jws.HandlerChain;
import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebResult;
import javax.jws.WebService;
import javax.jws.soap.SOAPBinding;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.context.support.SpringBeanAutowiringSupport;

import com.ejie.ab18a.model.Ab18aAplicacionInvocante;
import com.ejie.ab18a.model.Ab18aConfiguracionAplicacion;
import com.ejie.ab18a.model.Ab18aErrorAplicacion;
import com.ejie.ab18a.model.Ab18aEsquemasXSD;
import com.ejie.ab18a.model.Ab18aEstadoPeticion;
import com.ejie.ab18a.model.Ab18aEstadoTramitacion;
import com.ejie.ab18a.model.Ab18aPeticionEnvio;
import com.ejie.ab18a.model.Ab18aTicketPeticion;
import com.ejie.ab18a.model.Ab18aTipoBoletin;
import com.ejie.ab18a.properties.Ab18aPropertiesManager;
import com.ejie.ab18a.service.Ab18aAplicacionInvocanteService;
import com.ejie.ab18a.service.Ab18aConfiguracionAplicacionService;
import com.ejie.ab18a.service.Ab18aErrorAplicacionService;
import com.ejie.ab18a.service.Ab18aEsquemasXSDService;
import com.ejie.ab18a.service.Ab18aEstadoPeticionService;
import com.ejie.ab18a.service.Ab18aEstadoTramitacionService;
import com.ejie.ab18a.service.Ab18aKomunesService;
import com.ejie.ab18a.service.Ab18aPeticionEnvioService;
import com.ejie.ab18a.service.Ab18aTicketPeticionService;
import com.ejie.ab18a.service.Ab18aTipoBoletinService;
import com.ejie.ab18a.utils.Constantes;
import com.ejie.ab18a.utils.Utilidades;
import com.ejie.ab18a.ws.model.ErrorWS;
import com.ejie.ab18a.ws.model.PeticionEnvioRequest;
import com.ejie.ab18a.ws.model.PeticionEnvioResponse;
import com.ejie.ab18a.ws.model.TicketResponse;

/**
 * @author XXXX
 * 
 */
@WebService(serviceName = "envioBoletinWS", portName = "envioBoletinWSPort", targetNamespace = "http://com.ejie.ab18a.webservice")
@SOAPBinding(parameterStyle = SOAPBinding.ParameterStyle.WRAPPED)
@HandlerChain(file = "server-handlers.xml")
public class PeticionWebServiceImpl extends SpringBeanAutowiringSupport {

	private static final Logger logger = LoggerFactory.getLogger(PeticionWebServiceImpl.class);

	@Autowired()
	private Ab18aTipoBoletinService tipoBoletinService;

	@Autowired()
	private Ab18aAplicacionInvocanteService aplicacionInvocanteService;

	@Autowired()
	private Ab18aPeticionEnvioService peticionEnvioService;

	@Autowired()
	private Ab18aTicketPeticionService ticketPeticionService;

	@Autowired()
	private Ab18aConfiguracionAplicacionService configuracionAplicacionService;

	@Autowired()
	private Ab18aEsquemasXSDService esquemasXSDService;

	@Autowired()
	private Ab18aKomunesService komunesService;

	@Autowired()
	private Ab18aEstadoTramitacionService estadoTramitacionService;

	@Autowired()
	private Ab18aEstadoPeticionService estadoPeticionService;

	@Autowired()
	private Ab18aErrorAplicacionService errorAplicacionService;

	/**
	 * @author XXXX
	 * 
	 */
	@WebMethod()
	public PeticionEnvioResponse recibirPeticion(PeticionEnvioRequest peticionEnvioRequest) {

		// Boolean para controlar si pasa las validaciones generales
		boolean validado = true;
		// Boolean para controlar si ha habido alguna excepción durante el
		// proceso de validación
		boolean exception = false;
		boolean ticketValido = true;

		InputStream streamXML = null;

		List<ErrorWS> listaErrores = new ArrayList<ErrorWS>();

		PeticionEnvioResponse response = new PeticionEnvioResponse();

		Ab18aPeticionEnvio peticionEnvio = new Ab18aPeticionEnvio();

		Ab18aAplicacionInvocante aplicacionInvocante = null;

		Ab18aConfiguracionAplicacion configuracionAplicacion = new Ab18aConfiguracionAplicacion();

		try {
			peticionEnvio = this.rellenarPeticion(peticionEnvio, peticionEnvioRequest);

			// Validaciones generales
			if (Utilidades.nullOrZero(peticionEnvioRequest.getIdBoletin())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_1, null));
			} else {
				Ab18aTipoBoletin tipoBoletin = new Ab18aTipoBoletin();
				tipoBoletin.setT06IdBoletin(Long.valueOf(peticionEnvioRequest.getIdBoletin()));

				// Buscamos el tipo de boletin en BBDD
				tipoBoletin = this.tipoBoletinService.find(tipoBoletin);

				// Comprobamos que exista el tipo de boletin recibido
				if (Utilidades.nullOrEmpty(tipoBoletin)) {
					// Si no existe no pasa la validación
					validado = false;
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_2, null));
				}
			}

			// Comprobamos que no sea null la aplicación invocante
			if (Utilidades.nullOrZero(peticionEnvioRequest.getIdOrigenAplicacion())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_3, null));
			} else {
				aplicacionInvocante = new Ab18aAplicacionInvocante();

				// Aplicacion invocante a encontrar
				aplicacionInvocante.setT03IdAplicacion(peticionEnvioRequest.getIdOrigenAplicacion());

				aplicacionInvocante = this.aplicacionInvocanteService.find(aplicacionInvocante);

				if (Utilidades.nullOrEmpty(aplicacionInvocante)) {
					// Si no existe no pasa la validación
					validado = false;
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_4, null));
				}
			}

			if (Utilidades.nullOrZero(peticionEnvioRequest.getIdOrigen())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_31, null));
			} else {
				if (peticionEnvioRequest.getIdOrigen().toString().length() > Constantes.LONGITUD_MAX_ID_ORIGEN) {
					validado = false;
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_32, null));
				}
			}

			if (Utilidades.notNullAndNotZero(peticionEnvioRequest.getTipoEnvio())) {
				if (StringUtils.isNumeric(peticionEnvioRequest.getTipoEnvio().toString())) {
					if (peticionEnvioRequest.getTipoEnvio().compareTo(Constantes.TIPO_ENVIO_DIRECTO_ID) == 0) {
						peticionEnvio.setT02TipoEnvio(Constantes.TIPO_ENVIO_DIRECTO_ID);
					} else if (peticionEnvioRequest.getTipoEnvio().compareTo(Constantes.TIPO_ENVIO_PROGRAMADO_ID) == 0) {
						if (Utilidades.nullOrEmpty(peticionEnvioRequest.getFechaProg())) {
							validado = false;
							listaErrores.add(this.obtenerError(Constantes.ERROR_ID_5, null));
						}
						peticionEnvio.setT02TipoEnvio(Constantes.TIPO_ENVIO_PROGRAMADO_ID);
					} else {
						validado = false;
						listaErrores.add(this.obtenerError(Constantes.ERROR_ID_6, null));
					}
				} else {
					validado = false;
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_7, null));
				}
			} else {
				peticionEnvio.setT02TipoEnvio(Constantes.TIPO_ENVIO_DIRECTO_ID);
			}

			if (Utilidades.nullOrEmpty(peticionEnvioRequest.getAnuncio())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_8, null));
			} else {
				if (Utilidades.notNullAndNotEmpty(aplicacionInvocante)
						&& aplicacionInvocante.getT03ModoRecepcion().compareTo(Constantes.MODO_RECEPCION_XML) == 0) {
					String strXML = new String(Base64.decodeBase64(peticionEnvioRequest.getAnuncio()), "UTF-8");
					peticionEnvio.setT01AnuncioXml(strXML);
					peticionEnvio.setT01AnuncioXmlConvertido(strXML);
					streamXML = new ByteArrayInputStream(new String(Base64.decodeBase64(peticionEnvioRequest.getAnuncio())).getBytes());
					String cabecera = strXML.substring(1, 5);
					if (cabecera.indexOf("xml") < 0) {
						validado = false;
						listaErrores.add(this.obtenerError(Constantes.ERROR_ID_9, null));
					}
				} else {
					// Modo de recepción por defecto ZIP
					byte[] data = Base64.decodeBase64(peticionEnvioRequest.getAnuncio());

					peticionEnvio.setT01AnuncioZip(data);

					ZipInputStream zipStream = new ZipInputStream(new ByteArrayInputStream(data));

					ByteArrayOutputStream out = null;

					while (Utilidades.notNullAndNotEmpty(zipStream.getNextEntry())) {
						out = this.obtenerContenidoZip(new ByteArrayOutputStream(), zipStream);
						zipStream.closeEntry();
					}

					if (Utilidades.notNullAndNotEmpty(out)) {
						streamXML = new ByteArrayInputStream(out.toByteArray());
						peticionEnvio.setT01AnuncioXml(new String(out.toByteArray(), "UTF-8"));
						peticionEnvio.setT01AnuncioXmlConvertido(new String(out.toByteArray(), "UTF-8"));
					} else {
						validado = false;
						listaErrores.add(this.obtenerError(Constantes.ERROR_ID_10, null));
					}
				}
			}

			// Recuperamos el valor configurado de la validación de komunes
			configuracionAplicacion = this.configuracionAplicacionService.find(Constantes.VALIDACION_GENERAL_KOMUNES);

			if (Utilidades.nullOrEmpty(peticionEnvioRequest.getIdPoder())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_11, null));
			}

			if (Utilidades.nullOrEmpty(peticionEnvioRequest.getIdEntidadImp())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_12, null));
			}

			if (Utilidades.nullOrEmpty(peticionEnvioRequest.getIdOrgano())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_13, null));
			}

			if (Utilidades.nullOrEmpty(peticionEnvioRequest.getIdEntidadTram())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_14, null));
			}

			if (Utilidades.nullOrEmpty(peticionEnvioRequest.getIdMesa())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_15, null));
			}

			// En caso de que haya que pasar la validacion de komunes
			if (Utilidades.notNullAndNotEmpty(configuracionAplicacion) && configuracionAplicacion.getT00Valor()) {
				listaErrores.addAll(this.validarKomunes(peticionEnvioRequest));
				if (!listaErrores.isEmpty()) {
					validado = false;
				}
			}

			if (Utilidades.nullOrEmpty(peticionEnvioRequest.getEstadoTram()) || !StringUtils.isNumeric(peticionEnvioRequest.getEstadoTram())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_21, null));
			} else {
				Ab18aEstadoTramitacion estadoTramitacion = new Ab18aEstadoTramitacion(Integer.valueOf(peticionEnvioRequest.getEstadoTram()), "", "");
				estadoTramitacion = this.estadoTramitacionService.find(estadoTramitacion);

				if (Utilidades.nullOrEmpty(estadoTramitacion) || Utilidades.nullOrZero(estadoTramitacion.getIdEstadoTramitacion())) {
					validado = false;
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_22, null));
				}
			}

			if (Utilidades.nullOrEmpty(peticionEnvioRequest.getCodExpediente())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_23, null));
			} else {
				if (peticionEnvioRequest.getCodExpediente().length() > Constantes.LENGTH_EXP_CONTRATACION) {
					validado = false;
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_24, null));
				} else {
					peticionEnvio.setT01CodExpContratacion(peticionEnvioRequest.getCodExpediente());
				}
			}

			// Recuperamos el valor configurado de la validación específica
			configuracionAplicacion = this.configuracionAplicacionService.find(Constantes.VALIDACION_ESPECIFICA);

			// En caso de que haya que pasar la validacion específica
			if (Utilidades.notNullAndNotEmpty(configuracionAplicacion) && configuracionAplicacion.getT00Valor()
					&& Utilidades.notNullAndNotEmpty(streamXML)) {
				configuracionAplicacion = this.configuracionAplicacionService.find(Constantes.VALIDACION_ACTUALIZAR_ESQUEMAS);

				List<String> directorios = comprobarActualizacionEsquemas(configuracionAplicacion, peticionEnvioRequest);

				if (Utilidades.notNullAndNotEmpty(directorios)) {
					if (!validarEsquemas(streamXML, peticionEnvioRequest.getIdBoletin())) {
						validado = false;
						listaErrores.add(this.obtenerError(Constantes.ERROR_ID_27, null));
					}
				} else {
					validado = false;
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_36, null));
				}
			}
		} catch (Exception e) {
			exception = true;
			listaErrores.add(new ErrorWS("00999", ExceptionUtils.getStackTrace(e)));
			logger.error("PeticionWebServiceImpl: recibirPeticion() - ", e);
		} finally {

			Integer idEstado = obtenerEstado(exception, validado);

			if (Utilidades.nullOrZero(peticionEnvioRequest.getTicket())) {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_25, null));
			} else {
				if (isTicketInvalido(peticionEnvioRequest)) {
					validado = false;
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_26, null));
					ticketValido = false;
				}
			}

			peticionEnvio.setT11IdEstado(idEstado);

			Ab18aPeticionEnvio peticionAux = new Ab18aPeticionEnvio();

			if (Utilidades.notNullAndNotEmpty(peticionEnvioRequest.getTicket())
					&& Utilidades.notNullAndNotEmpty(peticionEnvioRequest.getIdOrigenAplicacion())) {
				try {
					peticionAux.setT03IdOrigen(peticionEnvioRequest.getIdOrigenAplicacion());
					peticionAux.setT07IdTicket(peticionEnvioRequest.getTicket());
					peticionAux = this.peticionEnvioService.findIfExists(peticionAux);
				} catch (Exception e) {
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_28, null));
					logger.error("PeticionWebServiceImpl: recibirPeticion() - Ocurrió un error al buscar la petición: " + e.getMessage());
				}
			}

			if (Utilidades.nullOrEmpty(peticionAux) || Utilidades.nullOrEmpty(peticionAux.getT01IdPeticionEnvio())) {
				if (isInsertable(peticionEnvio)) {
					try {
						if (ticketValido) {
							peticionEnvio = this.peticionEnvioService.add(peticionEnvio);
							Ab18aEstadoPeticion estadoPeticion = new Ab18aEstadoPeticion(Long.valueOf(peticionEnvio.getT11IdEstado()));
							estadoPeticion = this.estadoPeticionService.find(estadoPeticion);
							response = this.generarRespuesta(peticionEnvio, estadoPeticion);
						}
					} catch (Exception e) {
						ErrorWS errorAux = this.obtenerError(Constantes.ERROR_ID_29, null);
						errorAux.setDescripcion(errorAux.getDescripcion() + " " + e);
						listaErrores.add(errorAux);
						logger.error("PeticionWebServiceImpl: recibirPeticion() - Ocurrió un error al insertar la petición: " + e.getMessage());
					}
				}
			} else {
				validado = false;
				listaErrores.add(this.obtenerError(Constantes.ERROR_ID_30, null));
			}
		}

		response.setErrores(listaErrores);

		return response;
	}

	/**
	 * Función para validar los campos komunes de la petición de envío
	 * 
	 * @param peticionEnvioRequest
	 *            Petición de envío
	 * @return Listado de errores. En caso de estar vacia, es que no ha habido
	 *         ningún error durante la ejecución
	 */
	private List<ErrorWS> validarKomunes(final PeticionEnvioRequest peticionEnvioRequest) {

		List<ErrorWS> listaErrores = new ArrayList<ErrorWS>();

		if (Utilidades.notNullAndNotEmpty(peticionEnvioRequest.getIdPoder()) && Utilidades.notNullAndNotEmpty(peticionEnvioRequest.getIdEntidadImp())
				&& Utilidades.notNullAndNotEmpty(peticionEnvioRequest.getIdOrgano())) {
			try {
				if (!this.komunesService.comprobarPoderEntidadOrganoCoherente(peticionEnvioRequest.getIdPoder(),
						peticionEnvioRequest.getIdEntidadImp(), peticionEnvioRequest.getIdOrgano())) {
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_20, peticionEnvioRequest));
					PeticionWebServiceImpl.logger.error("No existe el organo: " + listaErrores);
				}
			} catch (Exception e) {
				PeticionWebServiceImpl.logger.error("Error al comprobar la coherencia del poder, entidad impulsora y organo: " + e);
			}
		}

		if (Utilidades.notNullAndNotEmpty(peticionEnvioRequest.getIdEntidadTram()) && Utilidades.notNullAndNotEmpty(peticionEnvioRequest.getIdMesa())) {
			try {
				if (!this.komunesService.comprobarEntidadMesaCoherente(peticionEnvioRequest.getIdEntidadTram(), peticionEnvioRequest.getIdMesa())) {
					listaErrores.add(this.obtenerError(Constantes.ERROR_ID_18, peticionEnvioRequest));
				}
			} catch (Exception e) {
				PeticionWebServiceImpl.logger.error("Error al comprobar la coherencia entre la entidad tramitadora y la mesa de contratacion: " + e);
			}
		}

		return listaErrores;
	}

	@WebMethod
	public @WebResult(name = "ticketResponse", partName = "ticketResponse", targetNamespace = "")
	TicketResponse obtenerTicket(@WebParam(name = "idAplicacionOrigen") Integer idAplicacionOrigen) {
		TicketResponse ticketRespuesta = new TicketResponse();

		Ab18aAplicacionInvocante aplicacionInvocante = new Ab18aAplicacionInvocante(idAplicacionOrigen);
		aplicacionInvocante = this.aplicacionInvocanteService.find(aplicacionInvocante);

		if (aplicacionInvocante != null && !aplicacionInvocante.getT03DescEs().equals("")) {

			Ab18aTicketPeticion ticketPeticion = new Ab18aTicketPeticion();
			ticketPeticion.setT03IdAplicacion(aplicacionInvocante.getT03IdAplicacion());
			ticketPeticion.setT07FechaPeticion(Calendar.getInstance().getTime());
			ticketPeticion.setT07Ticket("Ticket Prueba");
			try {
				ticketPeticion = this.ticketPeticionService.add(ticketPeticion);
			} catch (Exception e) {
				logger.error("PeticionWebServiceImpl: recibirPeticion() - Ocurrió un error al generar el ticket.", e);
				ticketRespuesta.setError("Ocurrió un error al generar el ticket");
			}

			ticketRespuesta.setIdTicket(ticketPeticion.getT07IdTicket());
			ticketRespuesta.setIdAplicacionOrigen(ticketPeticion.getT03IdAplicacion());
			ticketRespuesta.setFechaPeticion(ticketPeticion.getT07FechaPeticion());

		} else {
			ticketRespuesta.setError("No existe la aplicación proporcionada");
			PeticionWebServiceImpl.logger.error("Error: NO existe la aplicacion");
		}

		return ticketRespuesta;
	}

	/**
	 * Crea los XSD comunes necesarios para validar mediante el XSD de
	 * formulario correspondiente el XML enviado en el ZIP
	 * 
	 * @throws IOException
	 * @throws SQLException
	 */
	private void crearXSD(Integer idBoletin) throws IOException, SQLException {

		// Cargamos la lista de esquemas
		Ab18aEsquemasXSD esquemaAux = new Ab18aEsquemasXSD();
		esquemaAux.setT06IdBoletin(idBoletin);
		try {
			List<Ab18aEsquemasXSD> listaEsquemas = this.esquemasXSDService.findAll(esquemaAux, null);

			if (Utilidades.notNullAndNotEmpty(listaEsquemas)) {
				InputStream streamXSD = null;
				for (Ab18aEsquemasXSD esquema : listaEsquemas) {

					streamXSD = IOUtils.toInputStream(esquema.getT04Esquema());
					File fichero = new File(esquema.getT04ruta() + esquema.getT04CodEsquema());
					fichero.getParentFile().mkdirs();
					OutputStream outStream = new FileOutputStream(fichero);

					byte[] buffer = new byte[8 * 1024];

					int bytesRead;

					while ((bytesRead = streamXSD.read(buffer)) != -1) {
						outStream.write(buffer, 0, bytesRead);
					}

					IOUtils.closeQuietly(streamXSD);
					IOUtils.closeQuietly(outStream);
				}
			} else {
				PeticionWebServiceImpl.logger.debug("PeticionWebServiceImpl: crearXSD() - No se han encontrado esquemas");
			}
		} catch (IOException e) {
			PeticionWebServiceImpl.logger.error("PeticionWebServiceImpl: recibirPeticion() - Error al obtener los bytes del esquema, causa: "
					+ e.getMessage());
			throw e;
		}
	}

	private Ab18aPeticionEnvio rellenarPeticion(Ab18aPeticionEnvio peticionEnvio, PeticionEnvioRequest peticionEnvioRequest) {

		// ID boletin a insertar en BBDD
		peticionEnvio.setT06IdBoletin(peticionEnvioRequest.getIdBoletin());
		// Aplicacion invocante a insertar en BBDD
		peticionEnvio.setT03IdOrigen(peticionEnvioRequest.getIdOrigenAplicacion());
		peticionEnvio.setT01IdPoder(peticionEnvioRequest.getIdPoder());
		peticionEnvio.setT01IdMesa(peticionEnvioRequest.getIdMesa());
		peticionEnvio.setT01IdEntidadImp(peticionEnvioRequest.getIdEntidadImp());
		peticionEnvio.setT01IdEntidadTramitadora(peticionEnvioRequest.getIdEntidadTram());
		peticionEnvio.setT01IdOrgano(peticionEnvioRequest.getIdOrgano());
		peticionEnvio.setT07IdTicket(peticionEnvioRequest.getTicket());
		peticionEnvio.setT01EstadoTramitacion(peticionEnvioRequest.getEstadoTram());
		peticionEnvio.setT01IdOrigen(peticionEnvioRequest.getIdOrigen());
		peticionEnvio.setT01Paralizada(Constantes.NUMERO_0);

		if (Utilidades.notNullAndNotZero(peticionEnvioRequest.getTipoEnvio())
				&& peticionEnvioRequest.getTipoEnvio().compareTo(Constantes.TIPO_ENVIO_PROGRAMADO_ID) == 0) {
			peticionEnvio.setT01FechaProgrEnvio(peticionEnvioRequest.getFechaProg());
		} else {
			peticionEnvio.setT01FechaProgrEnvio(Calendar.getInstance().getTime());
		}

		return peticionEnvio;
	}

	private ErrorWS obtenerError(Integer id, PeticionEnvioRequest peticionEnvioRequest) {
		Ab18aErrorAplicacion error = new Ab18aErrorAplicacion(id, "", "", "");
		error = this.errorAplicacionService.find(error);

		if (Utilidades.notNullAndNotZero(error.getId()) && (error.getId() >= Constantes.ERROR_ID_17 && error.getId() <= Constantes.ERROR_ID_20)) {
			if (Utilidades.notNullAndNotEmpty(error.getDescripcionError())) {
				String descError = error.getDescripcionError();
				descError = descError.replace("param1", String.valueOf(peticionEnvioRequest.getIdPoder()));
				if (error.getId() >= Constantes.ERROR_ID_17 && error.getId() <= Constantes.ERROR_ID_18) {
					descError = descError.replace("param2", String.valueOf(peticionEnvioRequest.getIdEntidadTram()));
					if (error.getId().compareTo(Constantes.ERROR_ID_18) == 0) {
						descError = descError.replace("param3", String.valueOf(peticionEnvioRequest.getIdMesa()));
					}
				} else {
					descError = descError.replace("param2", String.valueOf(peticionEnvioRequest.getIdEntidadImp()));
					if (error.getId().compareTo(Constantes.ERROR_ID_20) == 0) {
						descError = descError.replace("param3", String.valueOf(peticionEnvioRequest.getIdOrgano()));
					}
				}
				error.setDescripcionError(descError);
			}
		}

		return new ErrorWS(error.getCodError(), error.getDescripcionError());
	}

	private ByteArrayOutputStream obtenerContenidoZip(ByteArrayOutputStream out, ZipInputStream zipStream) throws IOException {
		byte[] byteBuff = new byte[4096];
		int bytesRead = 0;
		while ((bytesRead = zipStream.read(byteBuff)) != -1) {
			out.write(byteBuff, 0, bytesRead);
		}

		out.close();

		return out;
	}

	private Integer obtenerEstado(Boolean exception, Boolean validado) {
		Integer idEstado;
		if (exception) {
			idEstado = Constantes.ESTADO_ERRONEA_RECEPCION;
		} else {
			if (validado) {
				idEstado = Constantes.ESTADO_PENDIENTE_ENVIO;
			} else {
				idEstado = Constantes.ESTADO_RECHAZADA_VALIDACION;
			}
		}

		return idEstado;
	}

	private PeticionEnvioResponse generarRespuesta(Ab18aPeticionEnvio peticionEnvio, Ab18aEstadoPeticion estadoPeticion) {
		PeticionEnvioResponse response = new PeticionEnvioResponse();

		response.setTicket(String.valueOf(peticionEnvio.getT01IdPeticionEnvio()));
		response.setEstado(estadoPeticion.getT11DescEstado());
		response.setCodEstado(estadoPeticion.getT11IdEstado().intValue());

		return response;
	}

	private Boolean isInsertable(Ab18aPeticionEnvio peticionEnvio) {
		if (Utilidades.notNullAndNotEmpty(peticionEnvio.getT02TipoEnvio()) && Utilidades.notNullAndNotEmpty(peticionEnvio.getT06IdBoletin())
				&& Utilidades.notNullAndNotEmpty(peticionEnvio.getT11IdEstado()) && Utilidades.notNullAndNotEmpty(peticionEnvio.getT07IdTicket())
				&& Utilidades.notNullAndNotEmpty(peticionEnvio.getT03IdOrigen()) && Utilidades.notNullAndNotEmpty(peticionEnvio.getT01IdOrgano())
				&& Utilidades.notNullAndNotEmpty(peticionEnvio.getT01IdOrigen())) {
			return true;
		} else {
			return false;
		}
	}

	private Boolean isTicketInvalido(PeticionEnvioRequest peticionEnvioRequest) {
		Ab18aTicketPeticion ticket = new Ab18aTicketPeticion();
		ticket.setT07IdTicket(peticionEnvioRequest.getTicket());

		ticket = this.ticketPeticionService.find(ticket);

		if (Utilidades.nullOrEmpty(ticket)
				|| Utilidades.nullOrZero(peticionEnvioRequest.getIdOrigenAplicacion())
				&& ticket.getT03IdAplicacion().compareTo(
						Utilidades.nullOrZero(peticionEnvioRequest.getIdOrigenAplicacion()) ? 0 : peticionEnvioRequest.getIdOrigenAplicacion()) != 0) {
			return true;
		} else {
			return false;
		}
	}

	private Boolean validarEsquemas(InputStream streamXML, Integer idBoletin) throws IOException {

		boolean resultado = false;
		String nombreXSD = "";
		int contador = 0;
		Ab18aEsquemasXSD esquema = new Ab18aEsquemasXSD();

		esquema.setT06IdBoletin(idBoletin);

		// Obtenemos la lista de las rutas con los esquemas
		List<String> listaRutas = this.esquemasXSDService.findRutas(esquema);

		// Comprobamos por cada ruta hasta que pase el
		if (Utilidades.notNullAndNotEmpty(listaRutas)) {
			while (!resultado && contador < listaRutas.size()) {
				streamXML.reset();
				if (listaRutas.get(contador).contains("2.0.8") && idBoletin.compareTo(Constantes.ID_BOLETIN_DOUE) == 0) {
					nombreXSD = Constantes.TED_ESENDER_XD;
				} else {
					nombreXSD = Constantes.TED_ESENDER_XSD;
				}
				resultado = Utilidades.validarContraXSD(streamXML, listaRutas.get(contador) + nombreXSD);
				contador++;
			}
		}

		return resultado;
	}

	private List<String> comprobarActualizacionEsquemas(Ab18aConfiguracionAplicacion configuracionAplicacion,
			PeticionEnvioRequest peticionEnvioRequest) throws IOException, SQLException {

		logger.debug("PeticionWebServiceImpl: comprobarActualizacionEsquemas() - Entramos");
		logger.debug("PeticionWebServiceImpl: comprobarActualizacionEsquemas() - filepath: " + Ab18aPropertiesManager.getProperty("file.path"));
		logger.debug("PeticionWebServiceImpl: comprobarActualizacionEsquemas() - actualizarEsquemas: " + configuracionAplicacion.getT00Valor());
		List<String> directorios = Utilidades.obtenerDirectorios(Ab18aPropertiesManager.getProperty("file.path"));

		// En caso de que haya que actualizar los esquemas
		if (Utilidades.nullOrEmpty(directorios) || (!directorios.contains("xsd"))
				|| (Utilidades.notNullAndNotEmpty(configuracionAplicacion) && configuracionAplicacion.getT00Valor())) {
			// Buscamos los XSD comunes a todos los formularios
			if (Utilidades.notNullAndNotEmpty(peticionEnvioRequest.getIdBoletin())) {
				this.crearXSD(peticionEnvioRequest.getIdBoletin());
			}

			directorios = Utilidades.obtenerDirectorios(Ab18aPropertiesManager.getProperty("file.path"));
		}

		if (Utilidades.notNullAndNotEmpty(directorios)) {
			logger.debug("PeticionWebServiceImpl: comprobarActualizacionEsquemas() - Directorios: " + directorios.size());
		}

		logger.debug("PeticionWebServiceImpl: comprobarActualizacionEsquemas() - Salimos");
		return directorios;
	}
}